;
;****************************************************************************
;
;  Purpose:
;	Interrupt Driven Serial Port Handler, External Memory Version
;
;  Date:
;	03/26/90
;
;  Author:
;	John C. Wren
;
;  Modications:
;	02/06/97 - Added Description Fields For Archive
;
;  Processor:
;	Generic 8031
;
;  Assembler:
;	Avocet AVA51
;
;  Dependencies:
;	Timer 1 Setup For Baud Rate Generation.  Timer 2 Can Be Used On
;	Processors Supporting It.
;
;  Files:
;	ADCAD.ASM
;
;  Comments:
;	A LJMP To SP0_INT Needs To Be Placed At Location 0023h.  
;
;  Philosophic:
;	The Equates In The Equates Section Should Be In The System 
;	EQUATES.INC File.
;
;****************************************************************************
;
;  Includes
;
;		%include "equates.inc"
		seg	code
;
;****************************************************************************
;
;  Publics
;
		public	SP0_INIT   
		public	SP0_INT    
		public	SP0_MCT
		public	SP0_MDT
		public	SP0_IN	   
		public	SP0_INS    
		public	SP0_INCLEAR
		public	SP0_OUT    
		public	SP0_OUTS
		public	SP0_OUTEMPTY
;
		extrn	UTIL_ADCAD
;
;****************************************************************************
;
;  Equates
;
ETX		equ	3			; ASCII ETX Character
;
;  These Are The Buffer Sizes, And Can Be Up To 255 Bytes
;
E_SP0IBL	equ	32			; 32 Character Input Buffer
E_SP0OBL	equ	16			; 16 Character Output Buffer
;
;****************************************************************************
;
;  Description:
;	Initialize Serial Port 0 Variables, Serial Port
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	None
;
;  Affected:
;	PSW.P, PSW.Z, Acc
;
;  Stack:
;	0 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	Sets Serial Port To Mode 1.  Timer 1 Or 2 (If Applicable) Should 
;	Already Be Setup In 8-Bit Auto Reload Mode For Baud Rate Generation.  
;	Must Be Called Before Any Serial I/O Performed.
;
SP0_INIT	proc
		clr	a			; 0
		mov	D_SP0IBH,a		; Set Input Buffer Header To 0
		mov	D_SP0IBT,a		; Set Input Buffer Tail To 0
		mov	D_SP0ICC,a		; Set Input Character Count To 0
		mov	D_SP0OBH,a		; Set Output Buffer Header To 0
		mov	D_SP0OBT,a		; Set Output Buffer Tail To 0
		mov	D_SP0OCC,a		; Set Output Character Count To 0
		clr	B_SP0OFL		; Set No Characters To Go Out
;
		mov	scon,#040h		; Set Serial Mode 1
		clr	ri			; Clear Receive Interrupt Flag
		setb	ti			; Pretend Character Sent OK
		setb	ren			; Allow Characters To Be Received
		setb	ps			; Set Serial As High Priority
		setb	es			; Allow Interrupts In Interrupt Structure
		ret				; Return To Caller
		endproc
;
;****************************************************************************
;
;  Description:
;	Output String From Code Area
;
;  Entry Requirements:
;	DPTR Points To ETX Terminated String In Code Space
;
;  On Exit:
;	None
;
;  Affected:
;	PSW.CY
;
;  Stack:
;	5 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_MCT 	proc
		push	dpl			; Save DPL
		push	dph			; Save DPH
		push	acc			; Save Acc
l?p1		clr	a			; Make A = 0 So MOVC Works
		movc	a,@a+dptr		; Get Character
		inc	dptr			; Point To Next Character
		cjne	a,#ETX,l?p2		; If Not EOM, Output Character
		pop	acc			; Recover Acc
		pop	dph			; Recover DPH
		pop	dpl			; Recover DPL
		ret				; Return To Caller
l?p2		call	SP0_OUT 		; Output Character
		sjmp	l?p1			; And Loop
		endproc
;
;****************************************************************************
;
;  Description:
;	Output String From External Memory Area
;
;  Entry Requirements:
;	DPTR Points To ETX Terminated String In External RAM 
;
;  On Exit:
;	None
;
;  Affected:
;	PSW.CY
;
;  Stack:
;	5 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_MDT 	proc
		push	dpl			; Save DPL
		push	dph			; Save DPH
		push	acc			; Save Acc
l?p1		movx	a,@dptr 		; Get Character
		inc	dptr			; Point To Next Character
		cjne	a,#ETX,l?p2		; If Not EOM, Output Character
		pop	acc			; Recover Acc
		pop	dph			; Recover DPH
		pop	dpl			; Recover DPL
		ret				; Return To Caller
l?p2		call	SP0_OUT 		; Output Character
		sjmp	l?p1			; And Loop
		endproc
;
;****************************************************************************
;
;  Description:
;	Interrupt Source Determiner
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	None
;
;  Affected:
;	None
;
;  Stack:
;	0 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	Normally, The RI Or TI Jump Should Always Be Taken.  On One 
;	Particular Emulator (That I'd Prefer Not To Mention Here), We've 
;	Seen It Generate A Serial Interrupt, With Neither RI Or TI Set.  
;	It Seems To Be Related To Transmitting, And I've Found That It
;	Works Best To Clear The Flag That Tells The Serial Transmit
;	Interrupt Routine Whether To Kick Start The Port By Setting TI
;	When Writing A Character.
;
SP0_INT 	proc
		jb	ri,SP0_IIN		; If Set, Receiver Interrupt
		jb	ti,SP0_IOUT		; If Set, Output Interrupt
		clr	B_SP0OFL		; If Weird Interrupt, Clear
		reti				; Neither?  (Should Never Happen!)
		endproc
;
;****************************************************************************
;
;  Description:
;	Process Interrupt Generated By Receiving A Character
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	None
;
;  Affected:
;	None
;
;  Stack:
;	6 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_IIN 	proc
		push	psw			; Save PSW
		push	acc			; Save Acc
		push	dpl			; Save DPL
		push	dph			; Save DPH
;
		mov	a,D_SP0ICC		; Get Input Character Count
		inc	a			; Add One
		cjne	a,#E_SP0IBL,l?p1	; If Not Same, Buffer Not Full
;
		sjmp	l?p3			; Return From Interrupt
;
l?p1		mov	a,D_SP0IBH		; Get End Of Buffer
		clr	c			; Clear Carry For Subtract
		subb	a,#E_SP0IBL		; Subtract Input Buffer Length
		mov	a,D_SP0IBH		; Get End Of Buffer
		jc	l?p2			; If Carry Set, Value Is In Range
		clr	a			; Point Back To Start Of Buffer
l?p2		mov	dptr,#X_SP0IBB		; Get Buffer Address
		call	UTIL_ADCAD		; Add 'A' To 'DPTR'
		inc	a			; Point To Next Header Location
		mov	D_SP0IBH,a		; Store Back
		mov	a,sbuf			; Get Character At Serial Port
		movx	@dptr,a			; Store Into Input Buffer
		inc	D_SP0ICC		; Increment Number Of Characters In Buffer
;
l?p3		pop	dph			; Recover DPH
		pop	dpl			; Recover DPL
		pop	acc			; Recover Acc
		pop	psw			; Recover PSW
		clr	ri			; Acknowledge We Got Character
		reti				; Return To Interrupted Routine
		endproc
;
;****************************************************************************
;
;  Description:
;	Process Interrupt Generated By Empty Transmitter
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	None
;
;  Affected:
;	None
;
;  Stack:
;	6 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_IOUT	proc
		push	psw			; Save PSW
		push	acc			; Save Acc
		push	dpl			; Save DPL
		push	dph			; Save DPH
;
		mov	a,D_SP0OCC		; Get Character Count
		jnz	l?p1			; If Character Count ~0, Stuff To Be Sent
		clr	B_SP0OFL		; Set Not Transmitting
		sjmp	l?p3			; Mark Interrupt As Done, And Return
;
l?p1		mov	a,D_SP0OBH		; Get End Of Buffer
		clr	c			; Clear Carry For Subtract
		subb	a,#E_SP0OBL		; Subtract Output Buffer Length
		mov	a,D_SP0OBH		; Get End Of Buffer
		jc	l?p2			; If Carry Set, Value Is In Range
		clr	a			; Point Back To Start Of Buffer
l?p2		mov	dptr,#X_SP0OBB		; Point To Buffer
		call	UTIL_ADCAD		; Add 'A' To 'DPTR'
		inc	a			; Point To Next Location
		mov	D_SP0OBH,a		; Store Back
		movx	a,@dptr			; Get Character
		mov	sbuf,a			; Get Character At Serial Port
		dec	D_SP0OCC		; Decrement Number Of Characters In Buffer
;
l?p3		pop	dph			; Recover DPH
		pop	dpl			; Recover DPL
		pop	acc			; Recover Acc
		pop	psw			; Recover PSW
		clr	ti			; Transmitter Interrupt Acknowledged
		reti				; Return To Interrupted Routine
		endproc
;
;****************************************************************************
;
;  Description:
;	Get A Character From The Serial Buffer, Waiting If None Present
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	Acc Contains Received Character
;
;  Affected:
;	PSW.CY, PSW.Z, PSW.P, Acc
;
;  Stack:
;	4 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	If Serial Interrupts Are Disabled (ES == 0), And A Character Is 
;	Present In The Buffer, Then The Serial Interrupt Will Be ReEnabled
;	On Exit.  If ES == 0 And No Characters Are In The Buffer, Then
;	The Routine Will Hang Waiting For A Serial Interrupt To Place A
;	Character Into The Buffer.
;
SP0_IN		proc
		call	SP0_INS 		; See If Character Pending
		jnc	SP0_IN			; Nope, Wait For One To Show Up
;
		push	dph			; Save DPH
		push	dpl			; Save DPL
		clr	es			; Disable Serial Interrupts
		mov	a,D_SP0IBT		; Get Tail
		push	acc			; Save Acc
		clr	c			; Clear Carry For Subtract
		subb	a,#E_SP0IBL		; Subtract Length
		pop	acc			; Recover Acc
		jc	l?p1			; If Carry, We Are In E_SP0IBL Length
		clr	a			; Point To Start of Buffer
l?p1		mov	dptr,#X_SP0IBB		; Point To Buffer
		call	UTIL_ADCAD		; Add 'A' To 'DPTR'
		inc	a			; Point To Next Location
		mov	D_SP0IBT,a		; Store Updated Tail Pointer
		movx	a,@dptr			; Fetch Character From Output Buffer
		dec	D_SP0ICC		; Decrement Number Of Characters In Buffer
		setb	es			; Re-Enable Serial Interrupts
		pop	dpl			; Recover DPL
		pop	dph			; Recover DPH
		ret				; Return It To Caller
		endproc
;
;****************************************************************************
;
;  Description:
;	Return Carry Clear If No Character Pending, Else Carry Set
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	CY == 1 If Character Present, CY == 0 If None
;
;  Affected:
;	PSW.CY
;
;  Stack:
;	1 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_INS 	proc
		push	acc			; Save Acc
		mov	a,D_SP0IBH		; Get Input Header
		cjne	a,D_SP0IBT,l?p1		; If Not Same, Character Pending
		clr	c			; Clear Carry, Because No Character Is Here
		pop	acc			; Recover Acc
		ret				; Return To Caller
;
l?p1		setb	c			; Set Carry Flag, Character Is Here
		pop	acc			; Recover Acc
		ret				; Return To Caller
		endproc
;
;****************************************************************************
;
;  Description:
;	Clears Any Character In The Receive Character Buffer
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	Acc = 0
;
;  Affected:
;	PSW.Z, PSW.P, Acc
;
;  Stack:
;	0 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	Probably Would Be A Good Idea To Disable Receive Interrupts While
;	Doing This.  An Interrupt In The Middle May Cause Problems.
;
SP0_INCLEAR	proc
		clr	a			; 0
		mov	D_SP0IBH,a		; Set Input Buffer Header To 0
		mov	D_SP0IBT,a		; Set Input Buffer Tail To 0
		mov	D_SP0ICC,a		; Set Input Character Count To 0
		ret				; Return To Caller
		endproc
;
;****************************************************************************
;
;  Description:
;	Output Character To Serial Buffer
;
;  Entry Requirements:
;	Acc Has Character To Write To Output Buffer
;
;  On Exit:
;	None
;
;  Affected:
;	PSW.CY
;
;  Stack:
;	5 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	If The Output Buffer Is Full, We Spin Waiting For A Serial Transmit
;	Interrupt To Free Up A Byte.  If Serial Interrupts Are Disabled
;	(ES == 0) When This Routine Is Called And There Is At Least One Byte
;	Avilable In The Buffer, Interrupts Will Be Reenabled.  If The Buffer 
;	Is Full, The Routine Will Hang Waiting For Space To Become Available
;	In The Buffer.
;
SP0_OUT 	proc
		push	dph			; Save DPH
		push	dpl			; Save DPL
		push	acc			; Save Acc
;		
l?p1		mov	a,D_SP0OCC		; Get Output Character Count
		inc	a			; Add 1
		cjne	a,#E_SP0OBL,l?p2	; If Not Same, Buffer Not Full
		sjmp	l?p1			; If Same, Wait Until Buffer Empties
;
l?p2		clr	es			; Disable Interrupts
;
		mov	a,D_SP0OBT		; Get End Of Buffer
		clr	c			; Clear Carry For Subtract
		subb	a,#E_SP0OBL		; Subtract Output Buffer Length
		mov	a,D_SP0OBT		; Get End Of Buffer
		jc	l?p3			; If Carry Set, Value Is In Range
		clr	a			; Point Back To Start Of Buffer
l?p3		mov	dptr,#X_SP0OBB		; Point To Buffer
		call	UTIL_ADCAD		; Add 'A' To 'DPTR'
		inc	a			; Next Buffer Location
		mov	D_SP0OBT,a		; Store Back
		pop	acc			; Get Character To Put In Buffer
		push	acc			; Resave Character
		movx	@dptr,a			; Store Character To Output Buffer
		inc	D_SP0OCC		; Increment Number Of Characters In Buffer
;
		jb	B_SP0OFL,l?p4		; If Output In Progress, Skip
		setb	B_SP0OFL		; Otherwise, Set Output In Progress Flag
		setb	ti			; Force Transmitter Interrupt
l?p4		setb	es			; Enable Interrupts
		pop	acc			; Recover Acc
		pop	dpl			; Recover DPL
		pop	dph			; Recover DPH
		ret				; Return To Caller
		endproc
;
;****************************************************************************
;
;  Description:
;	Return Carry Clear If Output Buffer Is Not Full, Else Carry Set
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	CY == 1 If Output Buffer Full, CY == 0 If Not
;
;  Affected:
;	PSW.CY, PSW.Z, PSW.P, Acc
;
;  Stack:
;	0 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_OUTS 	proc
		mov	a,D_SP0OCC		; Get Output Character Count
		inc	a			; Add 1
		cjne	a,#E_SP0OBL,l?p1	; If Not Same, Buffer Not Full
		setb	c			; CY == 1 Means Buffer Full
		ret				; Return To Caller
;
l?p1		clr	c			; CY == 0 Means Not Full
		ret				; Return To Caller
		endproc
;
;****************************************************************************
;
;  Description:
;	Return CY == 1 If Output Buffer Empty, Else CY == 0
;
;  Entry Requirements:
;	None
;
;  On Exit:
;	CY == 1 If Output Buffer Empty, CY == 0 If Not
;
;  Affected:
;	PSW.CY, PSW.Z, PSW.P, Acc
;
;  Stack:
;	0 Bytes, Not Including Space Used By Called Routines
;
;  Comments:
;	None
;
SP0_OUTEMPTY 	proc
		mov	a,D_SP0OCC		; Get Output Character Count
		jnz	l?p1			; If Not Empty
		setb	c			; CY ==	1 Means	Buffer Empty
		ret				; Return To Caller
;
l?p1		clr	c			; CY ==	0 Means	Not Empty
		ret				; Return To Caller
		endproc
;
;****************************************************************************
;
;  Data Area
;
		seg	bit
B_SP0OFL	ds	1			; Serial Output Flag
;
		seg	data
D_SP0IBH	ds	1			; Serial Input Head Pointer
D_SP0IBT	ds	1			; Serial Input Tail Pointer
D_SP0ICC	ds	1			; Serial Input Character Count
D_SP0OBH	ds	1			; Serial Output Buffer Head Pointer
D_SP0OBT	ds	1			; Serial Output Buffer Tail Pointer
D_SP0OCC	ds	1			; Serial Output Character Count

		seg	xdata
X_SP0IBB	ds	E_SP0IBL		; Serial Input Buffer
X_SP0OBB	ds	E_SP0OBL		; Serial Output Buffer
;
;*******************************************************************************
;
		end
